
#include "p33Fxxxx.h"
#include "Infrared.h"
#include "ir_codes.h"
#include "SDCard.h"
#include "ADC.h"
#include "DAC.h"
#include "LEDs.h"
#include "Timer.h"
#include "Playback.h"
#include "dsp.h"
#include "sine.h"
#include "config.h"
#include "dsp.h"
#include <string.h>

// Device configuration registers
_FGS(GWRP_OFF & GCP_OFF); // we don't want to write protect the flash
_FOSCSEL(FNOSC_FRC); // start up initially without the PLL but instead using the internal fast RC oscillator
_FOSC(FCKSM_CSECMD & OSCIOFNC_OFF & POSCMD_HS);
_FWDT(FWDTEN_OFF); // not using the watchdog timer
_FICD( ICS_PGD3&JTAGEN_OFF );

#define INIT_ATTEMPTS 3
#define INIT_DELAY    200

#define FFT_SIZE 1024
#define FFT_LOGN 10
#define FFT_WINDOW_HANNING

#if FFT_SIZE == 1024
#include "twidFactors_1024.h"
#elif FFT_SIZE == 512
#include "twidFactors_512.h"
#else
#error No twiddle factors for this FFT size!
#endif

fractcomplex FFTbuf[FFT_SIZE] __attribute__((space(ymemory), aligned(FFT_SIZE*4)));
unsigned short FFTbuf_pos;
unsigned short LEDChaser;
unsigned char LEDChaser_pos;

static unsigned short isqrt(unsigned short op) {
	unsigned short res, one;  

	res = 0;

	/* "one" starts at the highest power of four <= than the argument. */  
	one = 1 << 14;  /* second-to-top bit set */  
	while (one > op) one >>= 2;  
  
	while (one) {  
		if (op >= res + one) {  
			op -= res + one;  
			res += one << 1;  // <-- faster than 2 * one  
		}
		res >>= 1;  
		one >>= 2;  
	}  
	return res;  
}

static int compare_led_brightness(const void* a, const void* b) {
	return (int)((const led_bright*)a)->brightness - (int)((const led_bright*)b)->brightness;
}

/* Mode	Description
   0    16 frequency bands, 40Hz-20kHz, combining both channels (default)
   1    Two sets of 8 frequency bands, 40Hz-20kHz, one for each channel
   2    16 frequency bands, 40-1000Hz, combining both channels
   3    16 frequency bands, 1-20kHz, combining both channels
   4    16 frequency bands, 40Hz-20kHz, left channel only
   5    16 frequency bands, 40Hz-20kHz, right channel only
   6    16 frequency bands, 40-1000Hz, left channel only
   7    16 frequency bands, 1-20kHz, left channel only
   8    16 frequency bands, 40-1000Hz, right channel only
   9    16 frequency bands, 1-20kHz, right channel only */
unsigned char g_FFT_mode, g_FFT_channel;

unsigned char AGCenabled;
unsigned short AGC_average, actual_AGC_value = 0x1000;

static inline unsigned short min(unsigned long a, unsigned short b) {
	if( a > b )
		return b;
	else
		return a;
}

static inline signed short clip(signed long val) {
	if( val < -32768L )
		return -32768;
	else if( val > 32767L )
		return 32767;
	else
		return val;
}

static void AGC(signed short* data) {
	int i;
	unsigned short target_AGC_value;
	signed short dmin = 0, dmax = 0;
	for( i = 0; i < 256; ++i ) {
		if( data[i] < dmin )
			dmin = data[i];
		else if( data[i] > dmax )
			dmax = data[i];
	}
	if( -dmin > dmax )
		dmax = -dmin;

	AGC_average = (AGC_average * (0x10000UL-agc_filter_coeff) + dmax * (unsigned long)agc_filter_coeff) >> 16;
	if( dmax > AGC_average )
		AGC_average = dmax;
	target_AGC_value = min(AGC_average == 0 ? 0xFFFFFFFFUL : 0x07FFF000UL / AGC_average, agc_max);
	if( abs(target_AGC_value - actual_AGC_value) >= agc_max_error ) {
		if( target_AGC_value > actual_AGC_value ) {
			if( dmax >= 0x800 ) {	// don't want to keep increasing gain when it's silent
				if( actual_AGC_value > target_AGC_value - agc_delta_limit )
					actual_AGC_value = target_AGC_value;
				else
					actual_AGC_value += agc_delta_limit;
			}
		} else {
			if( actual_AGC_value < target_AGC_value + agc_delta_limit )
				actual_AGC_value = target_AGC_value;
			else
				actual_AGC_value -= agc_delta_limit;
		}
	}

	for( i = 0; i < 256; ++i )
		data[i] = clip((data[i] * (signed long)((unsigned long)actual_AGC_value))>>12);
}

void process_dac_data(signed short* data) {
	if( AGCenabled )
		AGC(data);

	if( FFTbuf_pos < FFT_SIZE ) {
		unsigned int i;
		if( g_FFT_mode == 0 || g_FFT_mode == 2 || g_FFT_mode == 3 ) {
			// both channels averaged
			for( i = 0; i < 128; i += 1 ) {
				signed short sample = (data[i*2+0] >> 1) + (data[i*2+1] >> 1);
#if defined(FFT_WINDOW_HANNING)
				FFTbuf[FFTbuf_pos].real = ((sample * (long)hanning[FFTbuf_pos < FFT_SIZE/2 ? FFTbuf_pos : FFT_SIZE - FFTbuf_pos - 1])) >> 16;
#elif defined(FFT_WINDOW_BLACKMAN)
				FFTbuf[FFTbuf_pos].real = ((sample * (long)blackman[FFTbuf_pos < FFT_SIZE/2 ? FFTbuf_pos : FFT_SIZE - FFTbuf_pos - 1])) >> 16;
#else
#error No valid FFT window defined!
#endif
				FFTbuf[FFTbuf_pos].imag = 0;
				++FFTbuf_pos;
			}
		} else if( g_FFT_mode == 4 || g_FFT_mode == 6 || g_FFT_mode == 7 || (g_FFT_mode == 1 && g_FFT_channel == 0) ) {
			// left channel only
			for( i = 0; i < 128; i += 1 ) {
				signed short sample = data[i*2+0];
#if defined(FFT_WINDOW_HANNING)
				FFTbuf[FFTbuf_pos].real = ((sample * (long)hanning[FFTbuf_pos < FFT_SIZE/2 ? FFTbuf_pos : FFT_SIZE - FFTbuf_pos - 1])) >> 16;
#elif defined(FFT_WINDOW_BLACKMAN)
				FFTbuf[FFTbuf_pos].real = ((sample * (long)blackman[FFTbuf_pos < FFT_SIZE/2 ? FFTbuf_pos : FFT_SIZE - FFTbuf_pos - 1])) >> 16;
#else
#error No valid FFT window defined!
#endif
				FFTbuf[FFTbuf_pos].imag = 0;
				++FFTbuf_pos;
			}
		} else if( g_FFT_mode == 5 || g_FFT_mode == 8 || g_FFT_mode == 9 || (g_FFT_mode == 1 && g_FFT_channel == 1) ) {
			// right channel only
			for( i = 0; i < 128; i += 1 ) {
				signed short sample = data[i*2+1];
#if defined(FFT_WINDOW_HANNING)
				FFTbuf[FFTbuf_pos].real = ((sample * (long)hanning[FFTbuf_pos < FFT_SIZE/2 ? FFTbuf_pos : FFT_SIZE - FFTbuf_pos - 1])) >> 16;
#elif defined(FFT_WINDOW_BLACKMAN)
				FFTbuf[FFTbuf_pos].real = ((sample * (long)blackman[FFTbuf_pos < FFT_SIZE/2 ? FFTbuf_pos : FFT_SIZE - FFTbuf_pos - 1])) >> 16;
#else
#error No valid FFT window defined!
#endif
				FFTbuf[FFTbuf_pos].imag = 0;
				++FFTbuf_pos;
			}
		}
	}
}

led_bright new_led_brightness[16], temp_led_brightness[16];

static void update_leds() {
	if( FFTbuf_pos == FFT_SIZE ) {
		unsigned int j, curband, nextband, min_inc, add, shift, numavg, brightness, first, last;
		unsigned short total_brightness = 0;

		FFTComplexIP(FFT_LOGN, FFTbuf, (fractcomplex*)__builtin_psvoffset(twidFactors), __builtin_psvpage(twidFactors));
		BitReverseComplex(FFT_LOGN, FFTbuf);

/* Mode	Description
   0    16 frequency bands, 40Hz-20kHz, combining both channels (default)
   1    Two sets of 8 frequency bands, 40Hz-20kHz, one for each channel
   2    16 frequency bands, 40-1000Hz, combining both channels
   3    16 frequency bands, 1-20kHz, combining both channels
   4    16 frequency bands, 40Hz-20kHz, left channel only
   5    16 frequency bands, 40Hz-20kHz, right channel only
   6    16 frequency bands, 40-1000Hz, left channel only
   7    16 frequency bands, 1-20kHz, left channel only
   8    16 frequency bands, 40-1000Hz, right channel only
   9    16 frequency bands, 1-20kHz, right channel only */

		first = 0;
		last = 16;
		if( g_FFT_mode == 0 || g_FFT_mode == 4 || g_FFT_mode == 5 ) {
			// full frequency span
			curband = 0;
			nextband = 1;
			min_inc = 4;
			add = 1;
			shift = 2;
		} else if( g_FFT_mode == 2 || g_FFT_mode == 6 || g_FFT_mode == 8 ) {
			// low frequencies only
			curband = 0;
			nextband = 1;
			min_inc = 20;
			add = 0;
			shift = 0;
		} else if( g_FFT_mode == 3 || g_FFT_mode == 7 || g_FFT_mode == 9 ) {
			// high frequencies only
			curband = 16;
			nextband = 18;
			min_inc = 0;
			add = 1;
			shift = 3;
		} else {
			// full frequency span but update only half the LEDs
			curband = 0;
			nextband = 2;
			min_inc = 0;
			add = 2;
			shift = 1;
			if( g_FFT_channel == 0 )
				last = 8;
			else
				first = 8;

			g_FFT_channel ^= 1;
		}

		for( j = first; j < last; ++j ) {
			numavg = 0;
			brightness = 0;
			while( curband < nextband ) {
				unsigned short val = ((long)FFTbuf[curband].real*(long)FFTbuf[curband].real + (long)FFTbuf[curband].imag*(long)FFTbuf[curband].imag) >> (16 - FFT_LOGN);
				brightness += isqrt( val );
				++curband;
				++numavg;
			}
			brightness /= numavg;
			new_led_brightness[j].index = j^1;
			if( brightness > new_led_brightness[j].brightness ) {
				if( brightness - new_led_brightness[j].brightness > spectrum_attack )
					new_led_brightness[j].brightness += spectrum_attack;
				else
					new_led_brightness[j].brightness = brightness;
			} else {
				if( new_led_brightness[j].brightness - brightness > spectrum_decay )
					new_led_brightness[j].brightness -= spectrum_decay;
				else
					new_led_brightness[j].brightness = brightness;
			}
			total_brightness += brightness;

			if( nextband < min_inc )
				++nextband;
			else
				nextband += (nextband+add) >> shift;
		}

		if( total_brightness < 16 ) {
			if( LEDChaser == 1406 || ++LEDChaser == 1406 ) { // 30 second delay
				for( j = 0; j < 16; ++j )
					new_led_brightness[j].brightness = sine[(unsigned char)(LEDChaser_pos + (15 - j) * 16)];
				++LEDChaser_pos;
			}
		} else {
			LEDChaser = 0;
			LEDChaser_pos = 0;
		}

		memcpy(temp_led_brightness, new_led_brightness, sizeof(temp_led_brightness));
		for( j = 0; j < 16; ++j )
			if( temp_led_brightness[j].brightness < spectrum_min_brightness )
				temp_led_brightness[j].brightness = 0;
		qsort(temp_led_brightness, 16, sizeof(*temp_led_brightness), &compare_led_brightness);
		asm("disi #32");
		memcpy(led_brightness, temp_led_brightness, sizeof(led_brightness));
		asm("disi #0");

		FFTbuf_pos = 0;
	}
}

unsigned char memoryCardSystemWasUp, memoryCardSystemIsUp;

void Standby() {
	unsigned int i, timer = 0;

	sdcard_set_power(0);
	leds_off();
	dac_stop_playback();
	adc_stop();

    __builtin_write_OSCCONH(0x02); // switch to crystal oscillator
    __builtin_write_OSCCONL(0x01);

	memoryCardSystemIsUp = memoryCardSystemUp;
	while( !memoryCardSystemUp || memoryCardSystemIsUp == memoryCardSystemUp ) {
		unsigned long ir_code = ir_final_code;
		ir_final_code = 0;
		if( ir_code ) {
			const char* fn = get_remote_control_function(ir_code);
			if( fn && !(ir_code&IR_REPEAT) && !strcmp(fn, "standby") )
				break;
		}
		LATAbits.LATA0 = ((++timer)&15) ? 1 : 0;
		memoryCardSystemIsUp = memoryCardSystemUp;
		memoryCardSystemUp = sdcard_get_presence();
	}
	LATAbits.LATA0 = 0;

    __builtin_write_OSCCONH(0x03); // switch back to PLL
    __builtin_write_OSCCONL(0x01);
	adc_init();
	for( i = 0; i < 16; ++i )
		led_brightness[i].brightness = 0;
	leds_init();
	dac_set_sample_rate(48000);

	memoryCardSystemWasUp = 0;
	memoryCardSystemIsUp = 0;
}

static void printushortdec(char* dest, unsigned short num) {
	unsigned char i;
	dest += 4;
	for( i = 5; i > 0; --i ) {
		dest[0] = num ? '0'+(num%10) : ' ';
		--dest;
		num /= 10;
	}
}

static void printushorthex(char* dest, unsigned short num) {
	unsigned char i;
	dest += 3;
	for( i = 4; i > 0; --i ) {
		unsigned char dig = num&15;
		dest[0] = dig < 10 ? '0'+dig : 'A'+dig-10;
		--dest;
		num >>= 4;
	}
}

extern unsigned char infrared_logging;
unsigned char infrared_log_open;
unsigned short ir_log_update_timer;
FIL ir_log;
extern unsigned short g_volume;

int main(void) {
	unsigned int ir_timer = 0, flash_timer = 0, num_flashes = 0;
	unsigned long ir_code;
	unsigned short cycles = 0;

	// enable output controlling LED1 (power/ack LED)
	TRISAbits.TRISA0 = 0;
	LATAbits.LATA0 = 1;

	Timer_Init();
	Infrared_Setup();
	restore_default_remote_codes();
	reset_config_to_default();
	sdcard_init();
	dac_init();
	leds_init();
	adc_init();
	dac_set_sample_rate(48000);
	AGCenabled = 1;

	LATAbits.LATA0 = 0;

	while(1) {
		++cycles;

		memoryCardSystemIsUp = memoryCardSystemUp;
		if( !memoryCardSystemWasUp && memoryCardSystemIsUp ) {
			unsigned int i;

			restore_default_remote_codes();
			reset_config_to_default();
			LEDChaser = 0;
			asm("disi #32");
			for( i = 0; i < 16; ++i )
				led_brightness[i].brightness = 0;
			asm("disi #0");
			AGCenabled = 0;

			if( !dac_is_running() )
				dac_start_playback();
			sdcard_set_power(1);
			Playback_Reset();
			if( Playback_Setup() == 0 ) {
				if( find_and_read_config() && start_playback_automatically )
					Playback_Start();
			} else {
				find_and_read_config();
			}
			flash_timer = Playback_Get_Status() == playing || Playback_Get_Status() == paused ? 64 : 48000;
			num_flashes = 8;
		} else if( memoryCardSystemWasUp && !memoryCardSystemIsUp ) {
			LEDChaser = 0;

			Playback_Reset();
			dac_set_sample_rate(48000);
			AGCenabled = 1;
		}
		memoryCardSystemWasUp = memoryCardSystemIsUp;

		if( memoryCardSystemIsUp && Playback_Get_Status() != error ) {
			Playback_Process();
		} else if( !memoryCardSystemIsUp || Playback_Get_Error() == no_wav_files ) {
			signed short* adc_data = adc_get_data();
			if( adc_data ) {
				dac_pause(false);
				dac_write(adc_data, 2, g_volume);
				if( dac_get_free_buffers() == 0 ) {
					if( !dac_is_running() )
						dac_start_playback();
				}
			}
		}
		update_leds(1);

		if( ir_timer && !--ir_timer ) {
			LATAbits.LATA0 ^= 1;
		}
		if( flash_timer && !--flash_timer ) {
			LATAbits.LATA0 ^= 1;
			if( --num_flashes )
				flash_timer = Playback_Get_Status() == playing || Playback_Get_Status() == paused ? 64 : 48000;
		}
		if( ir_log_update_timer && !--ir_log_update_timer ) {
			if( infrared_log_open ) {
				f_close(&ir_log);
				infrared_log_open = 0;
			}
		}

		ir_code = ir_final_code;
		ir_final_code = 0;
		if( ir_code ) {
			const char* fn = get_remote_control_function(ir_code);
			unsigned char valid = 0;
			if( fn && (!(ir_code&IR_REPEAT) || !strcmp(fn, "volup") || !strcmp(fn, "voldn") || !strcmp(fn, "back") || !strcmp(fn, "forward")) ) {
				valid = 1;
				if( !strcmp(fn, "volup") ) {
					playback_volume_up();
				} else if( !strcmp(fn, "voldn") ) {
					playback_volume_down();
				} else if( !strcmp(fn, "play") ) {
					Playback_Start();
				} else if( !strcmp(fn, "stop") ) {
					int i;
					Playback_Stop();
					for( i = 0; i < 16; ++i )
						led_brightness[i].brightness = 0;
				} else if( !strcmp(fn, "pause") ) {
					if( Playback_Get_Status() == paused )
						Playback_Start();
					else
						Playback_Pause();
				} else if( !strcmp(fn, "next_folder") ) {
					Playback_Stop();
					Playback_Next_Folder();
					Playback_Start();
				} else if( !strcmp(fn, "prev_folder") ) {
					Playback_Stop();
					Playback_Prev_Folder();
					Playback_Start();
				} else if( !strcmp(fn, "next_file") ) {
					Playback_Next_Track(1);
				} else if( !strcmp(fn, "prev_file") ) {
					Playback_Prev_Track(1);
				} else if( !strcmp(fn, "mute") ) {
					playback_toggle_mute();
				} else if( !strcmp(fn, "next_mode") ) {
					if( g_FFT_mode == 9 )
						g_FFT_mode = 0;
					else
						++g_FFT_mode;
				} else if( !strcmp(fn, "prev_mode") ) {
					if( g_FFT_mode == 0 )
						g_FFT_mode = 9;
					else
						--g_FFT_mode;
				} else if( fn[1] == '\0' && fn[0] >= '0' && fn[0] <= '9' ) {
					g_FFT_mode = fn[0] - '0';
				} else if( !strcmp(fn, "forward") ) {
					Playback_Forward();
				} else if( !strcmp(fn, "back") ) {
					Playback_Back();
				} else if( !strcmp(fn, "standby") ) {
					Standby();
				} else if( !strcmp(fn, "order") ) {
					if( Playback_Get_Order() == directory )
						Playback_Set_Order(sorted);
					else if( Playback_Get_Order() == sorted )
						Playback_Set_Order(shuffle);
					else
						Playback_Set_Order(directory);
				} else {
					valid = 0;
				}

				if( infrared_logging ) {
					unsigned int written;
					if( !infrared_log_open ) {
						char namebuf[32] = "0:\\IR_log.txt";
						memset(&ir_log, 0, sizeof(ir_log));
						if( f_open(&ir_log, namebuf, FA_OPEN_ALWAYS|FA_WRITE) == FR_OK ) {
							if( f_lseek(&ir_log, ir_log.fsize) == FR_OK ) {
								infrared_log_open = 1;
							} else {
								f_close(&ir_log);
							}
						}
					}
					if( infrared_log_open ) {
						char buf[20];
						printushortdec(buf, cycles);
						strcpy(buf+5, (ir_code&IR_RC5) ? " RC5(0x" : " NEC(0x");
						printushorthex(buf+12, ir_code);
						strcpy(buf+16, ")\r\n");
						f_write(&ir_log, buf, 19, &written);
						if( f_write(&ir_log, buf, 19, &written) != FR_OK || written != 19 ) {
							f_close(&ir_log);
							infrared_log_open = 0;
						} else {
							ir_log_update_timer = 256;
						}
					}
				}
			}
			if( valid || !fn ) {
				if( !ir_timer )
					LATAbits.LATA0 ^= 1;
				ir_timer = memoryCardSystemUp && (Playback_Get_Status() == playing || Playback_Get_Status() == paused) ? (valid ? 64 : 16) : (valid ? 48000 : 8192);
			}
		}
	}
}

int strncasecmp(const char* s1, const char* s2, size_t n) {
	while(n) {
		if( *s1 != *s2 )
			return (int)*s1 - (int)*s2;
		if( !*s1 )
			return 0;
		++s1;
		++s2;
		--n;
	}
	return 0;
}
